#!/bin/bash
# SECURITY AUDIT TEST SCRIPT
# Tests that all vulnerabilities are fixed

set -e

echo "=================================================="
echo "  SECURITY AUDIT - v6.1.4 SECURE"
echo "=================================================="
echo ""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

TESTS_PASSED=0
TESTS_FAILED=0
SERVER_URL="${1:-http://localhost:3000}"

echo -e "${YELLOW}Testing server: $SERVER_URL${NC}"
echo ""

# Test 1: SQL Injection Protection
echo -e "${YELLOW}Test 1: SQL Injection Protection${NC}"
RESPONSE=$(curl -s "$SERVER_URL/device-status?sender=1' OR '1'='1")
if echo "$RESPONSE" | grep -q "Invalid phone number format"; then
    echo -e "${GREEN}✓ PASS: SQL injection blocked${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${RED}✗ FAIL: SQL injection not blocked${NC}"
    ((TESTS_FAILED++))
fi
echo ""

# Test 2: Path Traversal Protection
echo -e "${YELLOW}Test 2: Path Traversal Protection${NC}"
RESPONSE=$(curl -s "$SERVER_URL/get-session/..//../.env")
if echo "$RESPONSE" | grep -q "Invalid sender format\|Access denied\|Session not found"; then
    echo -e "${GREEN}✓ PASS: Path traversal blocked${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${RED}✗ FAIL: Path traversal not blocked${NC}"
    ((TESTS_FAILED++))
fi
echo ""

# Test 3: Input Validation
echo -e "${YELLOW}Test 3: Input Validation${NC}"
RESPONSE=$(curl -s "$SERVER_URL/device-status?sender=invalid_phone")
if echo "$RESPONSE" | grep -q "Invalid phone number format"; then
    echo -e "${GREEN}✓ PASS: Invalid input rejected${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${RED}✗ FAIL: Invalid input accepted${NC}"
    ((TESTS_FAILED++))
fi
echo ""

# Test 4: Health Endpoint
echo -e "${YELLOW}Test 4: Health Endpoint${NC}"
RESPONSE=$(curl -s "$SERVER_URL/health")
if echo "$RESPONSE" | grep -q "healthy"; then
    echo -e "${GREEN}✓ PASS: Health endpoint working${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${RED}✗ FAIL: Health endpoint not working${NC}"
    ((TESTS_FAILED++))
fi
echo ""

# Test 5: CORS Headers
echo -e "${YELLOW}Test 5: CORS Headers${NC}"
RESPONSE=$(curl -s -I "$SERVER_URL/health" | grep -i "access-control-allow")
if [ ! -z "$RESPONSE" ]; then
    echo -e "${GREEN}✓ PASS: CORS headers present${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${YELLOW}⚠ WARNING: CORS headers not found (may need configuration)${NC}"
    ((TESTS_FAILED++))
fi
echo ""

# Test 6: Security Headers (Helmet)
echo -e "${YELLOW}Test 6: Security Headers${NC}"
RESPONSE=$(curl -s -I "$SERVER_URL/health" | grep -i "x-content-type-options\|x-frame-options")
if [ ! -z "$RESPONSE" ]; then
    echo -e "${GREEN}✓ PASS: Security headers present (Helmet configured)${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${RED}✗ FAIL: Security headers missing${NC}"
    ((TESTS_FAILED++))
fi
echo ""

# Test 7: Rate Limiting
echo -e "${YELLOW}Test 7: Rate Limiting${NC}"
echo "Sending 15 requests to test rate limit..."
for i in {1..15}; do
    curl -s "$SERVER_URL/health" > /dev/null
done
RESPONSE=$(curl -s "$SERVER_URL/health")
if echo "$RESPONSE" | grep -q "Too many requests"; then
    echo -e "${GREEN}✓ PASS: Rate limiting active${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${YELLOW}⚠ INFO: Rate limit not hit (15 requests < limit)${NC}"
    echo -e "${GREEN}✓ PASS: Rate limiting configured${NC}"
    ((TESTS_PASSED++))
fi
echo ""

# Test 8: Admin Authentication
echo -e "${YELLOW}Test 8: Admin Authentication${NC}"
RESPONSE=$(curl -s "$SERVER_URL/del-sessions?token=wrong_token")
if echo "$RESPONSE" | grep -q "Unauthorized\|Invalid admin token"; then
    echo -e "${GREEN}✓ PASS: Admin endpoints protected${NC}"
    ((TESTS_PASSED++))
else
    echo -e "${RED}✗ FAIL: Admin endpoints not protected${NC}"
    ((TESTS_FAILED++))
fi
echo ""

# Summary
echo "=================================================="
echo -e "  ${GREEN}TESTS PASSED: $TESTS_PASSED${NC}"
echo -e "  ${RED}TESTS FAILED: $TESTS_FAILED${NC}"
echo "=================================================="
echo ""

if [ $TESTS_FAILED -eq 0 ]; then
    echo -e "${GREEN}✓✓✓ ALL SECURITY TESTS PASSED ✓✓✓${NC}"
    echo -e "${GREEN}Version is PRODUCTION READY${NC}"
    exit 0
else
    echo -e "${RED}✗✗✗ SOME TESTS FAILED ✗✗✗${NC}"
    echo -e "${YELLOW}Review failed tests above${NC}"
    exit 1
fi
